from typing import List, Optional

from paddlex_hps_server import BaseTritonPythonModel, protocol, utils
from pydantic import BaseModel, Field
from typing_extensions import Annotated, TypeAlias


class InferenceParams(BaseModel):
    maxLongSide: Optional[Annotated[int, Field(gt=0)]] = None


class InferInput(BaseModel):
    image: str
    inferenceParams: Optional[InferenceParams] = None


Point: TypeAlias = Annotated[List[int], Field(min_length=2, max_length=2)]
BoundingBox: TypeAlias = Annotated[List[float], Field(min_length=4, max_length=4)]


class Table(BaseModel):
    bbox: BoundingBox
    html: str


class InferResult(BaseModel):
    tables: List[Table]
    layoutImage: str
    ocrImage: str


class TritonPythonModel(BaseTritonPythonModel):
    def _get_input_model_type(self):
        return InferInput

    def _get_result_model_type(self):
        return InferResult

    def _execute(self, input, log_id):
        if input.inferenceParams:
            max_long_side = input.inferenceParams.maxLongSide
            if max_long_side:
                return protocol.create_output_without_result(
                    422,
                    "`max_long_side` is currently not supported.",
                    log_id=log_id,
                )

        file_bytes = utils.get_raw_bytes(input.image)
        image = utils.image_bytes_to_array(file_bytes)

        result = list(self._pipeline.predict(image))[0]

        tables: List[Table] = []
        for item in result["table_result"]:
            tables.append(
                Table(
                    bbox=item["layout_bbox"],
                    html=item["html"],
                )
            )
        layout_image_base64 = utils.base64_encode(
            utils.image_to_bytes(result["layout_result"].img)
        )
        ocr_image_base64 = utils.base64_encode(
            utils.image_to_bytes(result["ocr_result"].img)
        )

        return InferResult(
            tables=tables, layoutImage=layout_image_base64, ocrImage=ocr_image_base64
        )
